﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.XR;

//CarController is responsible for the handling/physics of the car
public class CarController : MonoBehaviour
{
    //Public properties
    public float MaxSpeed = 3.0f;
    public float MaxReverseSpeed = 1.0f;
    public float Acceleration = 1.0f;
    public float Handling = 90.0f;
    public float DriftAmount = 0.0001f;
    public float DriftLowSpeedClamp = 0.8f;
    public float DriftHighSpeedClamp = 1.5f;
    public GameObject ButtonLegend;

    public float MinimumRPM = 0.0f;
    public float MaximumRPM = 1000.0f;
    public int MaximumGear = 5;

    public float CurrentRPM { get; private set; } = 0.0f;
    public int CurrentGear { get; private set; } = 1;
    public float SpeedGearMax { get; private set; }

    private Rigidbody2D _Body;  //The car uses the Rigidbody2D component for physics
    private string _EnteredColliderName;

    /// <summary>
    /// Gets the current speed of the car rounded to 2 d.p.
    /// </summary>
    /// <returns>The speed of the car to 2 d.p.</returns>
	public float GetCurrentSpeed()
	{
		return Convert.ToSingle(Math.Round(_Body.velocity.magnitude, 2));
	}

    /// <summary>
    /// Gets the velocity of the car in its local space
    /// </summary>
    /// <returns>The velocity of the car in its local space</returns>
    public Vector2 GetLocalVelocity()
    {
        return transform.InverseTransformDirection(_Body.velocity);
    }

	private void Start()
    {
        MaximumGear = 4;
        _Body = GetComponent<Rigidbody2D>();
        _Body.drag = 1.0f / MaxSpeed;
        SpeedGearMax = MaxSpeed / MaximumGear;
    }

    private void FixedUpdate()
    {
        float horizontalInput = Input.GetAxis("Horizontal");
        float verticalInput = Input.GetAxis("Vertical");

        DriftAmount = 0.65f;
        Handling = 10.0f;

        //Compute the speed of the car, add the force to the body
        Vector2 speed = transform.up * (verticalInput * Acceleration);
        _Body.AddForce(speed);

        _Body.velocity = Vector3.ClampMagnitude(_Body.velocity, MaxSpeed);  //Make sure we don't go faster than the maximum

        float direction = Vector2.Dot(_Body.velocity, _Body.GetRelativeVector(Vector2.up)); //Get the direction of the movement

        if (Acceleration > 0)
        {
            //We're accelerating, so let's apply rotation for turning
            float rotAmount = horizontalInput * Handling * ((MaxSpeed - _Body.velocity.magnitude + 0.1f) / MaxSpeed);

            if(_Body.velocity.magnitude <= 0.015f)
            {
                //We're going too slow to have effective turning
                rotAmount = 0.0f;
            }

            //Don't turn too much
            rotAmount = Mathf.Clamp(rotAmount, -5.0f, 5.0f);

			if (direction > 0)
            {
                _Body.rotation -= rotAmount;
            }

            else
            {
                _Body.rotation += rotAmount;
            }
        }

        float driftForce = 0.0f;

        //Now compute a drift force
        if (_Body.velocity.magnitude >= DriftHighSpeedClamp)
        {
            driftForce = Vector2.Dot(_Body.velocity, _Body.GetRelativeVector(Vector2.left)) * 10.0f;
        }

        else if(_Body.velocity.magnitude <= DriftLowSpeedClamp)
        {
            driftForce = Vector2.Dot(_Body.velocity, _Body.GetRelativeVector(Vector2.left)) * 4.0f;
        }

        else
        {
            driftForce = Vector2.Dot(_Body.velocity, _Body.GetRelativeVector(Vector2.left)) * DriftAmount;
        }

        //And apply the drift relative to the car's local movement
        Vector2 relativeForce = Vector2.right * driftForce;
        _Body.AddForce(_Body.GetRelativeVector(relativeForce));

        //Finally, let's make sure our car stays within the set speed boundaries
        float localVelocity = GetLocalVelocity().y;

        if (localVelocity > 0.0f)
        {
            if (_Body.velocity.magnitude > MaxSpeed)
            {
                _Body.velocity = _Body.velocity.normalized * MaxSpeed;
            }
        }

        else
        {
            if (_Body.velocity.magnitude > MaxReverseSpeed)
            {
                _Body.velocity = _Body.velocity.normalized * MaxReverseSpeed;
            }
        }

        //Make sure we don't fall through the world and always stay on the same Z for safety
        gameObject.transform.position = new Vector3
        (
            gameObject.transform.position.x,
            gameObject.transform.position.y,
            ConfigurationManager.Instance.Player.ZPosition - 10.0f
        );

        //Now compute the car RPM
        CurrentRPM = _Body.velocity.magnitude / (SpeedGearMax * CurrentGear);

        if(_Body.velocity.magnitude > (SpeedGearMax * CurrentGear))
        {
            //Above the speed for this gear, let's change
            AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles["GearShiftSFX"], false, Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume]));
            CurrentGear++;
        }

        else if(_Body.velocity.magnitude < (SpeedGearMax * (CurrentGear - 1)))
        {
            //Below the speed for this gear, let's change
            AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles["GearShiftSFX"], false, Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume]));
            CurrentGear--;
        }
    }

    /// <summary>
    /// Returns a boolean representing whether the trigger collision is with an interactable object
    /// </summary>
    /// <param name="collision">The collision we experienced</param>
    /// <returns>A boolean representing whether this trigger is interactable</returns>
    private bool IsTriggerInteractable(Collider2D collision)
    {
        if (Constants.GarageColliderNames.Contains(collision.name))
        {
            return true;    //It's a garage
        }

        else if(Constants.BuildingColliderNames.Contains(collision.name))
        {
            return collision.gameObject.transform.parent.GetComponent<BuildingController>().GetMission() != null;   //Check if this building has a mission - it's interactable if so
        }

        return false;
    }

    private void OnTriggerEnter2D(Collider2D collision)
    {
        if (!MissionsManager.Instance.IsOnMission && IsTriggerInteractable(collision))
        {
            //If we're not on a mission and have an interactable trigger, show the legend and store the collider name
            _EnteredColliderName = collision.name;
            ButtonLegend.SetActive(true);
        }

        else if(MissionsManager.Instance.IsOnMission)
        {
            //If we're on a mission, hide the legend
            ButtonLegend.SetActive(false);
        }
    }

    private void OnTriggerExit2D(Collider2D collision)
    {
        if (_EnteredColliderName == "" || _EnteredColliderName == collision.name)
        {
            //Hide the legend ONLY if we exit the trigger we entered
            _EnteredColliderName = "";
            ButtonLegend.SetActive(false);
        }
    }

    private void OnCollisionEnter2D(Collision2D collision)
    {
        if(collision.collider.gameObject.name.Contains("Road"))
        {
            //We collided with a road, determine our speed as a percentage of the maximum
            float speedPerc = collision.relativeVelocity.magnitude / MaxSpeed;

            if(speedPerc >= 0.25f && speedPerc < 0.50f)
            {
                //Small crash
                AudioManager.Instance.PlaySFX(AudioManager.Instance.AudioGroups["SmallCollisionsGroup"].GetRandomFile().Clip, false, Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume]));
            }

            else if (speedPerc >= 0.50f && speedPerc < 0.75f)
            {
                //Medium crash
                AudioManager.Instance.PlaySFX(AudioManager.Instance.AudioGroups["MediumCollisionsGroup"].GetRandomFile().Clip, false, Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume]));
            }

            else if(speedPerc >= 0.75f)
            {
                //Large crash
                AudioManager.Instance.PlaySFX(AudioManager.Instance.AudioGroups["LargeCollisionsGroup"].GetRandomFile().Clip, false, Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume]));
            }
        }
    }
}
